(function ($) {
	
	Pinboards.Comment = function (pinboard) {
		this.pinboard = pinboard;
		this.initialize();
	}
	
	Pinboards.Comment.prototype.closeDialog = function () {
		// This is independent of any pinboard (i.e., do a global search) so only one is ever active at a time
		$("#inline-dialog-pinboard-comment").remove();
	};
	
	Pinboards.Comment.prototype.comment = function (event, self) {
		var $commentButton = $(this);
		var $trigger = $commentButton.closest(".pinboard-tile").find(".inline-popup-trigger");
		
		self.closeDialog();

		AJS.InlineDialog(
			$trigger, 
			"pinboard-comment",
			function (content, trigger, showPopup) { self.initializeCommentDialog.call(this, content, trigger, showPopup, self) },
			{
				width: 480,
				gravity: "w",
				hideDelay: 3600000,
				offsetY: 100, 
				arrowOffsetY: -100,
				noBind: true
			}
		).show(event, $trigger);

		return false;
	};
	
	// Render a bit of wiki markup to place a reference to the current attachment into the comment.
	// This is called when the comment dialog opens to allow time for the server to convert the wiki markup,
	// which gets stashed in the dialog and appended to the user's comment when Save is pressed.
	Pinboards.Comment.prototype.fetchAttachmentDetails = function () {
		
		var self = this;
		
		// Clean any previously stashed attachment details
		// This is independent of any pinboard (i.e., do a global search) so only one is ever active at a time
		$("#inline-dialog-pinboard-comment .comment-extra").remove();
		
		// Bail if no attachment is involved
		if (!self.attachmentFilename) return;
		
		// Compose wiki markup to describe the attachment
		var markup = "{attachment-reference:attachment=" + self.attachmentFilename + "}";

		// Ask the server to create the editor XML from the wiki markup, and stash it in the dialog
        var conversionData = {
            wiki : markup,
            entityId :  self.targetPageId,
            spaceKey : self.spaceKey
        };
        $.ajax( {
            type : "POST",
            contentType : "application/json; charset=utf-8",
            url: Pinboards.contextPath + "/rest/tinymce/1/wikixhtmlconverter",
            data: $.toJSON(conversionData),
            dataType : "text", // if this is switched back to json be sure to use "text json". See CONFDEV-4799 for details.
            success : function(data) {
				var blob = $("<div>").addClass("comment-extra").css("display", "none").html(data);
				$("#inline-dialog-pinboard-comment").append(blob);
            },
            error: function(e){
                AJS.logError(e);
            },
            timeout: 7500
        });
	};
	
	Pinboards.Comment.prototype.initialize = function () {
		var self = this;

		// Listen for mouse-overs on the tiles
		self.pinboard.pinboard.on("mouseenter", ".pinboard-tile.unknown-comment", function () { 
			self.initializeTile.call(this, self);
		});
		
		// Listen for comment button clicks
		self.pinboard.pinboard.on("click", ".pinboard-comment-button", function (event) { self.comment.call(this, event, self); });
	};
	
	Pinboards.Comment.prototype.initializeCommentDialog = function (content, trigger, showPopup, self) {
		var $popup = $(this.popup);
		var $commentForm = $("#pinboard-comment-template").first().clone().removeAttr("id").show();
		content
			.css({ padding: "0px" })
			.html($("<div>").addClass("quick-comment-container").append($commentForm))
		.prepend($("<h2>").addClass("comment-heading").text(Pinboards.getText("com.brikit.pinboards.comment")));
		
		var $tile = $(trigger).closest(".pinboard-tile");
		var ceoId = $tile.data("id");
		self.targetPageId = ceoId;
		self.spaceKey = $tile.data("space-key");
		
		// Remember comment details
		self.commentParentId = 0;
		var commentOwnerId = $tile.data("comment-owner-id");
		if (commentOwnerId) {
			self.targetPageId = commentOwnerId;
			self.commentParentId = ceoId;
		}
		
		// Remember attachment details
		self.attachmentFilename = false;
		self.attachmentIsImage = false;
		var attachmentOwnerId = $tile.data("attachment-owner-id");
		if (attachmentOwnerId) {
			self.targetPageId = attachmentOwnerId;
			self.attachmentFilename = $tile.data("attachment-filename");
			self.attachmentIsImage = $tile.data("attachment-image");
		}
				
		var postInitialize = function (options) {
			content.find("#rte-toolbar .aui-toolbar2-secondary").remove();
			content.find(".rte-toolbar-group-link").removeClass("no-separator");
			
			AJS.Confluence.QuickEdit.QuickComment.postInitialise(options);

			// Do this after initialization to avoid unwanted scrolling by the comment editor
		    showPopup();
			
			// For attachments, get the embeddable blob for use in the comment
			self.fetchAttachmentDetails.call(self);
			
		}

        AJS.Confluence.QuickEdit.QuickComment.proceedWithActivation().done(function () {

			AJS.Confluence.QuickEdit.activateEditor({
                $container: content,
                $form: $commentForm,
                preInitialise: AJS.Confluence.QuickEdit.QuickComment.preInitialise,
                postInitialise: postInitialize,
                saveHandler: self.createSaveHandler(
					function (data) { self.saveHandler.call(self, data); },
					AJS.Confluence.QuickEdit.QuickComment.saveCommentErrorHandler,
					self),
                cancelHandler: AJS.Confluence.QuickEdit.QuickComment.cancelHandler,
                // Copied from the function in pre-Confluence 8:
                // plugins: AJS.Confluence.Editor._Profiles.createProfileForCommentEditor().plugins,
                plugins: "searchreplace confluenceimagedialog autocompletemacro confluencemacrobrowser confluenceleavecomment confluencewatch autoresize".split(" "),
                additionalResources: ["wrc!comment-editor"],
                timeoutResources: AJS.Confluence.QuickEdit.QuickComment.timeout,
				postDeactivate: function () { $popup.remove(); }
            });
        });

	};

	Pinboards.Comment.prototype.initializeTile = function (self) {
		var $button = $(".pinboard-comment-button", this);
		$button.tooltip({aria:true});
		$(this).removeClass("unknown-comment");
	};
	
    var generateUUID = function () {
        var s4 = function() {
            return Math.floor((1 + Math.random()) * 0x10000)
                    .toString(16)
                    .substring(1);
        };

        return s4() + s4() + '-' + s4() + '-' + s4() + '-' +
                s4() + '-' + s4() + s4() + s4();
    };
	
    /**
     * Create a save handler which is a function taking a single event that is to be called
     * when the save operation is activated on the editor.
     * 
     * @param successHandler the function to be called if save is successful. Takes a single argument which is the 
     * data returned from the save. 
     * @param errorHandler a function taking a single message parameter which is called if the save fails.
     * @return a function taking an event parameter which is suitable for use as a save handler for the editor
     */
    Pinboards.Comment.prototype.createSaveHandler = function(successHandler, errorHandler, self) {

        var uuid = generateUUID();

        // Replaces AJS.Rte.Content.isEmpty() in Confluence 8
        var isEmpty = function () {
            let e = tinymce.activeEditor.getContent().replace("&nbsp;","").replace("<br />","");
            return "<p></p>" === e.trim() || 0 === e.length;
        }

        return function(e) {

            e.preventDefault();

            if (isEmpty()) {
                // Removed for Confluence 8. If needed, replaced by AMD confluence-editor/notifications:
                // AJS.Confluence.EditorNotification.clear();
                // AJS.Confluence.EditorNotification.notify("warning", Pinboards.getText("content.empty"), 8);
                return;
            }

            var $form = AJS.Confluence.EditorLoader.getEditorForm();
            var captchaManager = new AJS.Confluence.QuickEditCaptchaManager($form);
            var changeCaptchaErrorHandler = function(message) {
                errorHandler(message);
                captchaManager.refreshCaptcha();
            };

			var watchPage = true;
			var $extraBlob = $("#inline-dialog-pinboard-comment .comment-extra");
			var extra = $extraBlob.length ? $extraBlob.html() : "";
			Confluence.Editor.CommentManager.saveComment(
				self.targetPageId, self.commentParentId, 
				tinymce.activeEditor.getContent() + extra,
            	watchPage, uuid, captchaManager.getCaptchaData(), successHandler, changeCaptchaErrorHandler);
        };
    };
	
	Pinboards.Comment.prototype.saveHandler = function (data) {
        var deferredCancel = AJS.Confluence.QuickEdit.QuickComment.cancelComment();
        deferredCancel.done(this.closeDialog);
	};

})(jQuery);
