(function ($) { 

	/*
		If provided, listener must implement:
			resultsLoaded(resultsData) - called when a search completes
				resultsData = { totalMatches: <count>, searchTime: <time> } when successful, or { error: "message" } when the search fails
				should return true if additional results should be loaded
        If provided, listener may implement:
            searchFormWillFireSearch() - called when the user makes a change to the search form, before the search is started
	*/
	TargetedSearch.SearchForm = function (element, searchResultsList, listener) {
		this.element = $(element);
		this.searchResultsList = searchResultsList;
		this.listener = listener;
		this.initialize();
		this.debug = false;
	};
	
	Object.assign(TargetedSearch.SearchForm.prototype, {
		
		addFilterToContainer: function (selector, name, id) {
			var $input = $("<input>")
				.attr("type", "hidden")
				.attr("name", name)
				.attr("value", id);
			this.filterContainer(selector).append($input);
		},

		emptyFilterContainer: function (selector) {
			return this.filterContainer(selector).empty();
		},
        
        emptySearchResults: function (spin) {
            this.searchResultsList.empty().spin();
        },

		filterContainer: function (selector) {
			return this.find(selector);
		},

		filterPickers: function () {
			return this.find(".filter-options, .space-category-options, .content-type-options, .date-range-options");
		},
	
		find: function (selector) {
			return selector ? $(selector, this.element) : this.element;
		},
		
		hasSearchCriteria: function () {
			return $.trim(this.queryString()).length || this.hasSelectedFilters();
		},
		
		hasSelectedFilters: function (hasFilters) {
			if (typeof(hasFilters) != "undefined") this.hasFilters = hasFilters;
			return this.hasFilters;
		},

		initialize: function () {
			var self = this;
			if (this.queryField().hasClass("live-search")) this.queryField().on("keyup", this.searchFieldKeyPressed.bind(this));
			if (!this.element.hasClass("aui-quicksearch")) this.find().submit(function() { return false; });
			
			if (TargetedSearch.spaceKey) this.find("input[name=hostSpaceKey]").val(TargetedSearch.spaceKey);
			if (TargetedSearch.pageId) this.find("input[name=pageId]").val(TargetedSearch.pageId);
		},
        
		moreResultsLoader: function () {
			return this.searchResultsList.data("more-results-loader");
		},

		removeFilter: function (e, selected) {
			e.preventDefault();
			this.removeFilterFromSelections(selected);
			this.searchCriteriaChanged();
		},
		
		removeFilterFromSelections: function (selected) {
			$selected = $(selected).addClass("removed");
			$selected.fadeOut(function () { $selected.remove(); });
		},

		queryField: function () {
            return this.find(".query-search-field");
		},
        
        querySubmitField: function () {
			return this.find("input[name=queryString]");
        },
	
		queryString: function () {
			return this.queryField().val();
		},
	
		queueSearch: function () {
	        clearTimeout(this.timer);
	        this.timer = setTimeout(this.searchCriteriaChanged.bind(this), 100);
		},

        // Returns false if a search doesn't fire as a result (no search criteria, repeating last search)
		searchCriteriaChanged: function (force) {
            if (this.searchInProgress) return this.queueSearch();
            
            if (this.listener && this.listener.searchFormWillFireSearch) this.listener.searchFormWillFireSearch();
      
			var loader = this.moreResultsLoader();
			var callback = this.listener ? this.listener.resultsLoaded.bind(this.listener) : null;

			if (!this.hasSearchCriteria()) {
                if (this.debug) console.log("No search criteria given. Reverting to help screen.");
                this.formValues = "";
				if (loader) loader.emptySearch(callback);
				return false;
			}

			if (this.debug) console.log("Search form values: " + this.find().serialize());
			if (loader) {
                let newQueryString = this.querySubmitField().val();
                let formValues = this.serialize();
                if (this.formValues == formValues && !force) {
                    if (this.debug) console.log('Already searched for', formValues);
                    return false;
                }
                this.formValues = formValues;
                this.searchInProgress = true;
                loader.restartSearch(formValues, newQueryString, function (results) {
                    if (this.debug) console.log("Search form, results loader callback received", newQueryString);
                    this.searchInProgress = false; 
                    this.lastQueryString = newQueryString; 
                    callback(results);
                    return true;
                }.bind(this));
			}
            
            return true;
		},
        
        searchFieldChanged: function () {
            this.searchCriteriaChanged();
        },
		
        searchFieldKeyPressed: function (e) {
            if (this.debug) console.log("searchFieldKeyPressed", e.key, e);
	        clearTimeout(this.searchFieldTimer);
	        this.searchFieldTimer = setTimeout(this.searchFieldChanged.bind(this), 500);
        },
		
		serialize: function () {
            // Copy the search terms to the submit-able input field, trimming white space if any
            this.querySubmitField().val($.trim(this.queryString()));
			return this.find().serialize();
		},
        
        startSearch: function () {
			var loader = this.moreResultsLoader();
            if (loader) loader.startSearch();
        },
		
        stopSearch: function () {
			var loader = this.moreResultsLoader();
            if (loader) loader.stopSearch();
        },
		
		updateFilterOption: function (filterOption) {
			this.find("input[name=filterOption]").val(filterOption);
		},
	
		updateSortBy: function (clause) {
			this.find("input[name=sortBy]").val(clause);
		}
		
	});
		
})(jQuery);
