(function ($) {
	
	Pinboards.debug = false;
	
	Pinboards.Pinboard = function (pinboard) {
		this.pinboard = $(pinboard);
		this.waitToResize = false;
		this.lastEntry = false;
		this.tilesInOrder = [];
		this.initialize();
	}
	
	Object.assign(Pinboards.Pinboard.prototype, {

		addContent: function (contents) {
			var self = this;
			
			self.loadMonitors().remove();

			var $contents = $(contents);

			// If no new tiles were found
			if (!$contents.length) {
				// If no results at all on the page, display a message indicating no results were found
				if (!self.tilesOnBoard().length) {
					self.shortestColumn().append(
						$("<div>").addClass("pinboard-tile").append(
							$("<div>").addClass("pinboard-content").append(
								$("<p>").append(
									$("<em>").append(Pinboards.getText("com.brikit.pinboards.no.results.for.this.pinboard"))
								)
							)
						)
					);
				}
			
				// If no more results, wait 10 seconds before triggering another search
				self.pinboard.addClass("out-of-content").removeClass("loading");
				self.addLoadMonitors();
				setTimeout(function () { self.pinboard.removeClass("out-of-content"); }, 10000);
				return;
			}

			// Neutralize the Theme Press element that pushes the footer to the bottom of the window
			$("#brikit-footer-shim").height(0);
		
			// Add each of the tiles to the pinboard
			$contents.each(function () {
				try {
					
					var $tile = $(this);
				
					Pinboards.logDebug("adding tile " + $tile.data("feed-entry"));
					self.tilesInOrder.push($tile);
					var $shortest = self.shortestColumn();
					$shortest.append($tile);

					// Adjust the height of the new tile's content area to the maximum allowed, fading out the content if it's too long
					if ($tile.height() >= self.maxContentHeight($tile)) {
						self.contentArea($tile).css({maxHeight: self.maxContentHeight($tile) + "px"});
						$tile.find(".fade-out").show();
					}
				
					// Add the tile height to the column
					self.columnHeight($shortest, self.columnHeight($shortest) + $tile.height());

					self.lastEntry = $tile;
					self.adjustImages($tile);

		            // Add share button bindings
		            // $shareButton = $tile.children('.pinboard-share-button');
		            // if ($shareButton.length > 0) Pinboards.Share.bindShareDialog($shareButton[0].id);
		
				}
				catch (err) {
					console.log("Error rendering tile: " + err);
					console.log(this);
				}
			});

			self.pinboard.removeClass("loading");

			self.addLoadMonitors();		
		},
	
		addLoadMonitors: function () {
			this.columns().each(function () {
				$(this).append($("<div>").addClass("load-monitor").append("<span class='aui-icon aui-icon-wait'></span>"));
			});
		},
	
		adjustColumns: function (firstTileOnPinboard) {
			var self = this;
			clearTimeout(self.resizeTimout);
			if (!firstTileOnPinboard && self.busy()) return self.resizeTimout = setTimeout(function () { self.adjustColumns.call(self) }, 200);
		
			// In Theme Press mobile view, let the tiles go full width
			if (Pinboards.isMobileView) {
				// self.pinboard.closest(".brikit-content-block").css({paddingLeft: 0, paddingRight: 0});
			}

			var windowWidth = self.pinboardContainer().width();

			var desiredColumnCount = Math.floor(windowWidth / self.calculatedTileWidth());
			if (desiredColumnCount < 1) desiredColumnCount = 1;
			var columnsChanged = self.columnCount() != desiredColumnCount;

			self.waitToResize = true;

			while (self.columnCount() < desiredColumnCount) {
				self.columns().last().after($("<div>").addClass("pinboard-column"));
			}
		
			while (self.columnCount() > desiredColumnCount) {
				self.columns().last().remove();
			}
		
			// Don't replace the tiles if we are just starting out
			if (!firstTileOnPinboard && columnsChanged) {
				self.tileWidth = undefined;
				self.tilesOnBoard().remove();
				self.columnHeight(self.columns(), 0);
				var allTiles = self.tilesInOrder.slice(0); // Use slice to create a copy
				self.tilesInOrder = [];
				self.addContent(allTiles);
			}

			self.waitToResize = false;
		
		},
		
		adjustImage: function ($image) {
			var imageWidth = $image.attr("width") || $image.width();
			var $tile = $image.closest(".pinboard-tile");
			var $content = $image.closest(".pinboard-content");
			if (imageWidth > $content.width()) {
				$image.removeAttr("width");
				var hPad = ($tile.innerWidth() - $content.width()) / 2;
				var vPad = ($content.innerHeight() - $content.height()) / 2;
				var css = { 
					maxWidth: "calc(100% + " + (hPad * 2) + "px)",
					marginLeft: -hPad + "px",
					height: "auto"
				};

				var $wrapper = $image.closest(".confluence-embedded-file-wrapper");
				if ($wrapper.length) {
					if (!$wrapper.closest(".pinboard-like").length && !$wrapper.closest(".pinboard-share").length &&
						!$wrapper.prev().length && !$wrapper.closest("p").prev().length) css["margin-top"] = -vPad + "px";
					if ($wrapper.closest("p").length) $wrapper.closest("p").addClass("image-wrapper");

				}
			
				$image.css(css);
			}
		},

		adjustImages: function (tile) {
			var self = this;
			$(".confluence-embedded-image:not(.width-adjusted)", tile).each(function() {
				var $image = $(this).addClass("width-adjusted");

				// Skip display of attachments: we're only interested in images rendered in pages and blogposts
				if ($image.closest(".pinboard-attachment").length) return;

				var image = new Image();
				image.onload = function () { 
					if (!$image.hasAttr("width")) $image.attr("width", image.width);
					self.adjustImage($image);
					self.respondToBrowserChange();
				}
				image.src = $image.attr("src");
			});
		},
	
		busy: function () {
			return this.waitToResize || this.isLoading();
		},
	
		// Return the tile width (in mobile view, it's the full window width)
		calculatedTileWidth: function () {
			if (typeof(this.tileWidth) == "undefined") {
                const widthParam = this.pinboard.data("tile-width");
				this.tileWidth =  Pinboards.isMobileView || !widthParam ? this.pinboardContainer().width() : widthParam;
				Pinboards.logDebug("calculated tile width: " + this.tileWidth);
				var macroId = this.pinboard.data("macro-id");
				var css = ".pinboard-macro-" + macroId + " .pinboard-column { width: " + this.tileWidth + "px; }";
				Pinboards.logDebug("calculated css: " + css);
				$(".pinboard-macro-style-" + macroId).text(css);
			}
			return this.tileWidth;
		},
	
		clickPageLink: function (event) {
			event.preventDefault();
			location.href = $(this).closest(".tile-wrapper").find(".page-link a").attr("href");
		},
	
		columnCount: function () {
			return this.columns().length;
		},
	
		columnHeight: function (column, height) {
			if (typeof(height) != "undefined") $(column).data("content-height", height);
			return $(column).data("content-height");
		},
	
		columns: function () {
			return this.pinboard.find("> .pinboard-columns > .pinboard-column");
		},
	
		// Return the content area for $tile
		contentArea: function ($tile) {
			return $tile.find(".pinboard-content");
		},
	
		documentationThemeContainer: function () {
			return $("#splitter-content");
		},

		entityData: function (button) {
			return {
					contentType: $(button).data("type"),
					contentId: $(button).data("id")
				}
		},
	
		initialize: function () {
		
			// Make sure we react to changes in the sidebar
			var self = this;
		    AJS.bind("sidebar.collapsed sidebar.expanded", function () {
				setTimeout(function () { self.respondToBrowserChange.call(self) }, 300);
			});
		
			// Listen for Theme Press structure changes
			if (typeof(ThemePress) != "undefined") ThemePress.Events.addCallback("structureDidChange navigatorDidChange", function () { 
				self.respondToBrowserChange.call(self);
			});
		
			// Listen for key window events
			$(window)
				.scroll(function () { self.respondToBrowserChange.call(self) })
				.resize(function () { self.respondToBrowserChange.call(self) });

			if (!self.pinboard.hasClass("disable-social-buttons")) {
				new Pinboards.Like(this);
				new Pinboards.Comment(this);
				// new Pinboards.Share(this);
			}
		
			// Handle clicks on tiles as if they were block links
			self.pinboard.on("click", ".pinboard-page .pinboard-content, .pinboard-page .pinboard-details", self.clickPageLink);

			// Special case for the Documentation Theme
			$(".theme-documentation #splitter-content").scroll(function () { self.respondToBrowserChange.call(self) });
			
			self.adjustColumns(true);

			self.load();
		
		},
	
		isLoading: function () {
			return this.pinboard.hasClass("loading") || this.pinboard.hasClass("out-of-content");
		},
	
		isUsingDocumentationTheme: function () {
			return this.documentationThemeContainer().length;
		},
	
		load: function () {
			Pinboards.logDebug("wants to load, busy = " + this.busy() + ", at " + (new Date().getTime() % 10000));
			if (this.busy()) return;
		
			this.pinboard.addClass("loading");
			var href = this.pinboard.attr("action");
			var data = {
				labels: this.pinboard.data("labels"),
				matchLabels: this.pinboard.data("match-labels"),
				excludeLabels: this.pinboard.data("exclude-labels"),
				order: this.pinboard.data("order"),
				includePages: this.pinboard.data("include-pages"),
				includeBlogposts: this.pinboard.data("include-blogposts"),
				includeFiles: this.pinboard.data("include-files"),
				includeSocial: this.pinboard.data("include-social"),
				showSpaceNames: this.pinboard.data("show-space-names"),
				showCreator: this.pinboard.data("show-creator"),
				showCreateDate: this.pinboard.data("show-create-date"),
				spaceFilter: this.pinboard.data("space-filter"),
				spaceKeysOrCategories: this.pinboard.data("space-keys-or-categories"),
				defaultSpaceKey: Pinboards.spaceKey,
                pageId: Pinboards.pageId
			};
            Pinboards.addToken(data);
			if (this.lastEntry) data.lastEntryString = this.lastEntry.data("feed-entry");
			if (this.pinboard.data("items-per-fetch")) data.itemsPerFetch = this.pinboard.data("items-per-fetch");
			Pinboards.logDebug("last entry " + data.lastEntryString);

			var self = this;
			$.get(href, 
				data, 
				function (response, textStatus, jqXHR) {
					self.pinboard.find(".waiting").remove();
					if ($(".errorMessage", response).length) {
						Pinboards.Dialog2.alert(response, {title: Pinboards.getText("com.brikit.pinboards.error.loading.pinboard.content")});
					}
					else {
						self.addContent($(response).find(".pinboard-tile"));
					}
				}
			);

		},
	
		loadMonitors: function () {
			return this.pinboard.find(".load-monitor");
		},
	
		// Return the maximum height of $tile's content area.
		maxContentHeight: function ($tile) {
			// If $tile specified a max height, use it, otherwise use the pinboard's default max height
			var tileMaxHeight = this.contentArea($tile).data("max-height");
			return tileMaxHeight ? tileMaxHeight : this.pinboard.data("tile-max-content-height");
		},

		// Return the element containing the pinboard
		pinboardContainer: function () {
			return this.pinboard.parent();
		},
	
		respondToBrowserChange: function () {

			var self = this;
			clearTimeout(self.respondWaitTimeout);
			if (self.busy()) return self.respondWaitTimeout = setTimeout(function () { self.respondToBrowserChange.call(self); }, 300);

			Pinboards.logDebug("responding to browser change, ensuring page is full: " + self.shouldFillPage());
			self.adjustColumns();
			var shouldLoad = self.loadMonitors().aboveTheFold(self.shouldFillPage() ? 0 : 25).length;
			Pinboards.logDebug("above the fold: " + shouldLoad);
			if (shouldLoad) self.load();
		},
	
		scrollingContainer: function () {
		    if (this.isUsingDocumentationTheme()) return this.documentationThemeContainer()
			return $((Pinboards.Browser.firefox || Pinboards.Browser.ie)? "html" : "body");
		},

		shortestColumn: function () {
			var $columns = this.columns();
			var $shortest = $columns.first();
			$columns.each(function (i, column) {
				if (!this.columnHeight(column)) this.columnHeight(column, 1);
				if (this.columnHeight(column) < this.columnHeight($shortest)) $shortest = $(column);
			}.bind(this));
			return $shortest;
		},
	
		shouldFillPage: function () {
	        return this.scrollingContainer().scrollTop() == 0;
		},
	
		tilesOnBoard: function () {
			return this.pinboard.find(".pinboard-tile");
		}

	});
			
	Pinboards.toFinalize(function ($) {
		Pinboards.isMobileView = $("body").hasClass("mobile");			

		$(".pinboard.pinboard-macro").each(function () {
			new Pinboards.Pinboard(this);
		})
	});

})(jQuery);
