(function ($) {
	
	ThemePress.PageTree = function (treeView) {
		this.treeView = $(treeView);
        this.ContentFlow = typeof(ContentFlow) != "undefined";
		this.debug = false;
		ThemePress.sidebarTreeView = this;

		var self = this;
		this.treeView.on("click", "li.unloaded, li:has(ul)", function (e) { self.expanderClicked(e, this); });

		this.ancestorExpansions = [
			"children.page",
			"ancestors",
			"metadata.properties.documentId",
            "metadata.properties.pageDocumentNumber",
            "metadata.properties.title",
			"ancestors.children.page",
			"ancestors.metadata.labels",
			"ancestors.children.page.metadata.labels",
			"ancestors.children.page.metadata.properties.documentId",
			"ancestors.children.page.metadata.properties.pageDocumentNumber",
			"ancestors.children.page.metadata.properties.title",
			"space",
			"space.homepage"
		].join("%2C");
		
        this.childrenExpansions = [
			"children.page",
			"metadata.labels",
			"metadata.properties.documentId",
			"metadata.properties.pageDocumentNumber",
			"metadata.properties.title",
            "children.page.metadata.properties.documentId",
            "children.page.metadata.properties.pageDocumentNumber",
            "children.page.metadata.properties.title"
		].join("%2C");
	};
	
	Object.assign(ThemePress.PageTree.prototype, {

        adjustOrderForDocumentIdentifiers: function ($container) {
            if (!this.ContentFlow || !$("body.document-identifiers-enabled").length) return;
            let needsReorder = false;
            let $elements = $container.children("li");
 			$.each($elements, function(index, page) { if ($(page).data("page-document-number")) needsReorder = true; });
            if (!needsReorder) return;
            $elements.sort(function(a, b) {
                return $(a).data("sortable-title").localeCompare($(b).data("sortable-title"));
            })
            .each(function() {
                $container.append(this);
            });
        },
        
        // page is the CQL content json object representing a page
        comparableContentFlowDocumentIdAndTitle: function (page) {
            let title = page.title;

            // If a leading period in the title, ensure that it sorts after all others
            title = (title.indexOf(".") == 0 ? "Z" : " ") + title;
            
            // The leading space ensures that pages with document ids will sort before other pages
            let pageNumber = ThemePress.contentFlowPageDocumentNumber(page);
            return (pageNumber ? " " + pageNumber + " " : "Z ") + title;
        },
        
		composeChildLevel: function (data, $ul, pageId) {
			this.treeView.spinStop();
			
			if (this.debug) console.log("loaded tree data", data);
	
			$.each(data.results, function(index, page) { this.composeNode(page, $ul); }.bind(this));
            this.adjustOrderForDocumentIdentifiers($ul);
			
			this.initializeTreeDepthMarkers();
			
			if (data._links.next) {
                this.fetchChildren(this.nextChunkUrl(data), function (data) { this.composeChildLevel(data, $ul, pageId) }.bind(this));
            }
			else {
                this.loadingChildren = false;
                this.initializeAfterLoad();
            }
		},
        
		composeListElement: function (page) {
            var titlePrefix = this.ContentFlow ? ThemePress.contentFlowDocumentId(page) : false;
			var $a = $("<a>").attr("href", ThemePress.contextPath + page._links.webui).text((titlePrefix ? titlePrefix + " " : "") + page.title);
			var $li = $("<li>").append($a);
			$.each(page.metadata.labels.results, function (i, label) { $li.addClass("label-" + label.name); });
			return $li;
		},
		
		composeNextLevel: function (data, $li) {
			if (this.debug) console.log("loaded children", data);
			
			$li.removeClass("unloaded").spinStop();
			var pageId = $li.data("page-id");
			
			this.composeChildLevel(data, $("ul", $li), pageId);
		},
		
		// page is the node we're composing for
		// $subTree is the LI element containing page's children
		composeNode: function (page, $subTree) {
			var children = page.children ? page.children.page.results : [];
			var hasChildren = children.length;
			var isAncestor = this.ancestors.filter(function (a) { return a.id == page.id }).length;
			var isCurrent = page.id == ThemePress.pageId;
			
			var $li = this.composeListElement(page)
				.attr("data-page-id", page.id)
				.adjustClass("collapsed", hasChildren)
				.adjustClass("leaf", !hasChildren)
				.adjustClass("ancestor", isAncestor)
				.adjustClass("current", isCurrent)
				.adjustClass("unloaded", !isAncestor)
                .data("sortable-title", this.comparableContentFlowDocumentIdAndTitle(page))
                .attr("data-sortable-title", this.comparableContentFlowDocumentIdAndTitle(page))
                .data("page-document-number", ThemePress.contentFlowPageDocumentNumber(page))
                .attr("data-page-document-number", ThemePress.contentFlowDocumentId(page))
                .attr("data-page-title-with-document-number", ThemePress.contentFlowPageTitleWithDocumentId(page));

			$subTree.append($li);
			
			// For the initial top-level drawing, if we're dealing with an ancestor, keep expanding tree down to the current page
			if (isAncestor && hasChildren) {
				var $ul = $("<ul>");
				$li.append($ul);

				var url = ThemePress.contextPath + "/rest/api/content/" + page.id + "/child/page?expand=" + this.childrenExpansions;
				if (this.debug) console.log(url);
				this.fetchChildren(url, function (data) { this.composeChildLevel(data, $ul, page.id); }.bind(this));
			}
			
            if (isCurrent) this.initializeAfterLoad();

			return $li;
		},
        
		composeTopLevel: function (data) {
			this.treeView.spinStop();
			
			if (this.debug) console.log("loaded tree data", data);
			
			$.each(data.results, function(index, page) { this.composeNode(page, this.topLevelUL); }.bind(this));
            this.adjustOrderForDocumentIdentifiers(this.topLevelUL);

			if (data._links.next) {
				this.loadTopLevelChunk(this.nextChunkUrl(data));
			}
			else {
				this.loadingTopLevel = false;
				if (!this.ancestors.length) this.initializeAfterLoad(); // Special case for top-level pages
			}
		},
		
		composeTree: function (results) {
			if (this.debug) console.log("loaded ancestors", results);
			
			// If no ancestors and not a top-level page, wait a bit and try again (newly created pages have a delay before ancestors are recognized)
			if (!results.ancestors.length && ThemePress.parentPageId) {
				setTimeout(this.fetchAncestorsOfCurrentPage.bind(this), 1000);
				return;
			}
			
			// Reverse the order so we can start at the leaf and build out
			this.ancestors = [];
			$.each(results.ancestors, function(index, ancestor) { this.ancestors.push(ancestor); }.bind(this));
			if (this.debug) console.log("ancestors", this.ancestors);
			
			this.loadTopLevel(this.ancestors[0] ? this.ancestors[0].id : results.space.homepage.id);
		},
		
		currentPage: function () {
			return this.find("li.current");
		},
		
		expanderClicked: function (event, target) {
			event.stopPropagation();
            
            this.stopAutoScrolling = true;
			
			var $target = $(target);
						
			// Only respond if clicking on the list element and if the click is in the expander icon zone
	        if (target != event.target || event.offsetX >= parseInt($target.children("a").first().css("margin-left"))) return true;
			
			$target.toggleClass('collapsed expanded');
			if ($target.hasClass("unloaded")) this.loadChildren(this.loadChildrenUrl($target.data("page-id")), $target);
			else $target.children('ul').toggle('fast');
			
		},
		
		fetchChildren: function (url, callback) {
			$.ajax({
				url: url,
				method: "GET",
				success: callback
			});
		},
		
		fetchAncestorsOfCurrentPage: function () {
			var url = ThemePress.contextPath + "/rest/api/content/" + ThemePress.pageId + "?expand=" + this.ancestorExpansions + "&limit=1&cqlcontext=%7B%7D";
			if (this.debug) console.log("Loading navigator:" + url);
			this.fetchChildren(url, this.composeTree.bind(this));
		},
		
		find: function (selector) {
			return selector ? $(selector, this.treeView) : this.treeView;
		},
	
		initializeAfterLoad: function (width) {
			// Expand to show the current page in the tree
			this.currentPage().parents("li.collapsed").toggleClass("collapsed expanded");
			
			this.initializeTreeDepthMarkers();
			
			if (!this.stopAutoScrolling) this.scrollToCurrentPage();
		},

		initializeTreeDepthMarkers: function () {
			// Add classes to indicate depth to the tree's ul elements
			this.find("ul").each(function () { 
                const depth = $(this).parents("ul").length;
                $(this).addClass(depth > 20 ? "too-deep" : "depth-" + depth);
            });
		},
		
		load: function () {
			if (ThemePress.isBlog) return;
			
			this.topLevelUL = $("<ul>");
			this.treeView.append(this.topLevelUL);
			
			this.fetchAncestorsOfCurrentPage();
		},
		
		loadChildren: function (url, $li) {
			if (this.debug) console.log("Loading children for:", $li, $li.data("page-id"));
			this.loadingChildren = true;

			var $ul = $li.find("ul");
			if (!$ul.length) {
				$ul = $("<ul>");
				$li.append($ul).spin();
			}
			
			this.fetchChildren(url, function (data) { this.composeNextLevel(data, $li); }.bind(this));
		},
		
		loadChildrenUrl: function (pageId) {
			return ThemePress.contextPath + "/rest/api/content/" + pageId + "/child/page?expand=" + this.childrenExpansions;
		},		
		
		loadTopLevel: function (pageId) {
			var url = this.loadChildrenUrl(pageId);
			if (this.debug) console.log(url);
			this.loadTopLevelChunk(url);
		},
		
		loadTopLevelChunk: function (url) {
			this.treeView.spin();
			this.loadingTopLevel = true;
			this.fetchChildren(url, this.composeTopLevel.bind(this));
		},
        		
        nextChunkUrl: function (data) {
            return data._links.context + data._links.next;
        },
        
		scrollToCurrentPage: function () {
			if (this.loadingTopLevel || this.loadingChildren) setTimeout(this.scrollToCurrentPage.bind(this), 250);
			else if (this.currentPage().get(0)) this.currentPage().get(0).scrollIntoView({block: "center", behavior: "smooth"});
		}

	});
	

	ThemePress.toInit(function ($) {
		$("#brikit-tree-view").each(function () { new ThemePress.PageTree(this); });
	});
	
})(jQuery);
