(function ($) {
	
	TargetedSearch.SearchField = function (element) {
		this.element = $(element);
		this.initialize();
		this.debug = false;
	};
	
	Object.assign(TargetedSearch.SearchField.prototype, {

		clearSearchResults: function (event) {
			// If we got here by clicking on the search terms or the results, do nothing
			if (event && $(event.target).closest(".targeted-search-field-query, .targeted-search-field-results").length) return;

			this.resultsContainer.empty();
		},
		
		displayQuickSearchResults: function ($results) {
			// Clear the wait indicator
			this.stopWaiting();

			if (this.debug) console.log("formatted results", $results);

			var $container = this.resultsContainer;

			// If it's inside a Theme Press block, make the block overflow so the popup is visible
			if ($("body.brikit.theme-base").length && $container.closest(".brikit-content-block").length) {
				$container.closest(".brikit-content-block").css("overflow", "visible");
				
				// var offset = $container.offset();
				// var width = $container.width();
				// var $mainContent = $("body");
				// var contentOffset = $mainContent.offset();
				// var top = offset.top - contentOffset.top;
				// var left = offset.left - contentOffset.left;
				// $mainContent.append($container);
				// $container.css({position: "absolute", top: top, left: left}).width(width);
			}

            // Line up search results with the search box
            var $sourceSearchField = this.searchTermsField;
            if ($sourceSearchField.length) {
                var left = $sourceSearchField.offset().left;
                var top = $sourceSearchField.offset().top + $sourceSearchField.outerHeight();
                $container.offset({ top: top, left: left});
            }

			// Add the results list to the DOM
	   		$container.html($results);
			
		},
		
		errorOccurred: function (x, s, t) {
			console.error(x);
			var data = { error: JSON.parse(x.responseText).message };
			console.error("An error occurred: " + data.error, x);
            this.stopWaiting();
		},
	
		find: function (selector) {
			return selector ? $(selector, this.element) : this.element;
		},

		formatResult: function (result) {
			var $li = $("<li class='with-space-name'>");
			
			var title = result.title;
			var link = TargetedSearch.contextPath + result.url;
			var $title = $("<a class='title'>")
				.attr("href", link)
				.attr("tooltip", title)
				.append($("<span class='icon aui-icon'>").addClass(result.iconCssClass))
				.append($("<span class='title-text'>").text(title));
			
			var $space = $("<a class='space-name'>")
				.attr("href", link);
				
			if (result.entityType == "content") $space.text(result.content.space ? result.content.space.name : "");
			else if (result.entityType == "space") $space.text("Space");

			return $li.append($title).append($space);
		},
		
		initialize: function () {
			this.form = this.find(".targeted-search-form");
			this.searchTermsField = this.find(".targeted-search-field-query").unbind();
            this.querySubmitField = this.find("input[name=queryString]");
			this.resultsContainer = this.find(".targeted-search-field-results");

			var self = this;
			this.find(".targeted-search-form").first().each(function () { self.searchForm = new TargetedSearch.SearchForm(this, self.find(".search-results-list"), self); });
			this.find(".selected-filters").first().each(function () { self.selectedFilters = new TargetedSearch.SelectedFilters(this, self.searchForm, self); });
			this.selectedFilters.updateSearchFiltersFromSelections();
			
			this.searchForm.find().submit(this.openSearchPanel.bind(this));
			
			// Ensure Confluence Sections don't chop off search results dropdowns
			this.element.closest(".innerCell").css({overflow: "visible"});
		
			// Input field and results bindings
			this.searchTermsField.keyup(this.searchCriteriaChanged.bind(this));
			this.searchTermsField.change(this.searchCriteriaChanged.bind(this));
			
			// SEARCH-686 -- the popup won't receive the first click when the input field has focus
			this.resultsContainer.mouseover(function (e) { this.searchTermsField.blur(); }.bind(this));

			// Close the search results when clicking elsewhere
			$(document).on("click", this.clearSearchResults.bind(this));

			// When refocusing on the search field, bring up 
			this.searchTermsField.focus(this.searchCriteriaChanged.bind(this));
			
			this.replaceConfluenceSearchField();
		},
		
		// Return whether a search is in progress
		isWaitingForSearch: function () {
			return this.form.data("waiting");
		},
	
		loadResults: function (data) {
			if (this.debug) console.log("search results", data);

			var $resultsList = $("<ol>");
			var $results = $("<div class='targeted-search-results-list aui-dropdown aui-dropdown-left'>").append($resultsList);
			
			if (!data.results.length) {
				var $empty = $("<li class='no-results-found'>").append($("<a>").append($("<span>").text(TargetedSearch.getText("com.brikit.targetedsearch.no.results.found"))));
				$resultsList.append($empty);
			}
			
			$.each(data.results, function (i, result) {
				$resultsList.append(this.formatResult(result));
			}.bind(this));
			
			this.displayQuickSearchResults($results);
		},
		
		openSearchPanel: function (event) {
			event.preventDefault();
			TargetedSearch.SearchPanel.openSearch(this.searchForm.serialize());
		},

		// Queue a delayed search, removing any previously queued (but not fired) search
		queueSearch: function () {
	        clearTimeout(this.timer);
	        this.timer = setTimeout(this.quickSearch.bind(this), 300);
		},
	
		// Fire quick search
		quickSearch: function () {
			// If a search is in progress, wait some more before firing the next search
			if (this.debug && this.isWaitingForSearch()) console.log("Search in progress. Waiting to fire search.");
			if (this.isWaitingForSearch()) return this.queueSearch();

			this.startWaiting();

			var serializedSearchParameters = this.searchForm.serialize();
			if (this.debug) console.log("query parameters: " + serializedSearchParameters);

            $.ajax({
                type: "POST",
                url: TargetedSearch.contextPath + "/plugins/targetedsearch/composecql.action",
                data: serializedSearchParameters,
                success: function (response) {
					if (this.debug) {
						console.log("Searching with CQL: " + TargetedSearch.cqlEncoded(response.cql));
						TargetedSearch.cqlDebugViewer().show().html(TargetedSearch.cqlEncoded(response.cql));
					}
					this.quickSearchQuery(response.cql);
				}.bind(this),
				error: this.errorOccurred.bind(this),
                dataType: "json"
            });
	    },
		
		quickSearchQuery: function (cql) {
			var data = {
				excerpt: "none",
				expand: "content.space",
				start: 0,
				limit: this.find("input[name=maxResults]").val() || 15
			};
			
			var cqlContext = {};
			if (TargetedSearch.spaceKey) cqlContext["spaceKey"] = TargetedSearch.spaceKey;
			if (TargetedSearch.pageId) cqlContext["contentId"] = TargetedSearch.pageId;
			data["cqlcontext"] = JSON.stringify(cqlContext);
			
			$.ajax({
			    url: TargetedSearch.contextPath + "/rest/api/search?cql=" + TargetedSearch.cqlEncoded(cql),
			    data : data,
			    type: "GET",
			    success: this.loadResults.bind(this),
				error: this.errorOccurred.bind(this),
			    dataType: "json"
			});
		},

		replaceConfluenceSearchField: function () {
			
			// If not configured to replace Confluence search, do nothing
			var $targetedSearch = this.find(".targeted-search-form[data-replace-default]").first();
			if (!$targetedSearch.length) return;

			// Block the default Confluence search from firing
			$("body:not(.mobile) #quick-search-query").unbind("focus").unbind("click");
			
			// Handle .brikit pages as a special case
			var isArchitectPage = (typeof(TargetedSearch.pageTitle) != "undefined") && (TargetedSearch.pageTitle.indexOf(".brikit") == 0);
			
			// If we're in a brikit layout page, append a placeholder for the macro we're about to move
			if (isArchitectPage && ThemePress) {
				$(".targeted-search-field[data-replace-default]", ThemePress.Edit.editableBlocks()).after("<p><em>" + TargetedSearch.getText("com.brikit.targetedsearch.replace-default-search.info") + "</em></p>");
			}
			
			var $quickSearch = $(".header-quicksearch");
			$targetedSearch.css({
				padding: $quickSearch.css("padding"),
				margin: $quickSearch.css("margin"),
				position: $quickSearch.css("position"),
				top: $quickSearch.css("top"),
				right: $quickSearch.css("right"),
				bottom: $quickSearch.css("bottom"),
				left: $quickSearch.css("left"),
				width: $quickSearch.css("width"),
				height: $quickSearch.css("height")
			});
			this.find(".targeted-search-field-query").css({
				width: "",
				maxWidth: ""
			});
			
			// Replace Confluence default search with targeted search field
            $targetedSearch.click(this.openSearchPanel.bind(this));
			$targetedSearch.attr("id","quick-search");
			$targetedSearch.removeAttr("style");
			$quickSearch.replaceWith($targetedSearch);
			$targetedSearch.show();
		},
		
		// Handle changing user input
		searchCriteriaChanged: function () {
            if (this.debug) console.log("Search criteria changed", this.searchTerms());
			if (!this.searchTerms()) this.clearSearchResults();
			else this.queueSearch();
		},

		searchTerms: function () {
			return this.searchTermsField.val();
		},
		
		// Add the search wait indicator
		startWaiting: function () {
            if (this.debug) console.log("Start search in progress");
			if (TargetedSearch.isConfluenceVersionAtLeast("7.0")) TargetedSearch.auiSpinner(this.find(".search-waiting"));
			else this.find(".search-waiting").spin();
			this.form.data("waiting", true);
		},
	
		// Remove the search wait indicator
		stopWaiting: function () {
            if (this.debug) console.log("Stop search in progress");
			if (TargetedSearch.isConfluenceVersionAtLeast("7.0")) this.find("aui-spinner").remove();
			else this.find(".search-waiting").spinStop();
			this.form.data("waiting", false);
		}

	});
	
	TargetedSearch.toFinalize(function ($) {
		$(".targeted-search-field").each(function () { new TargetedSearch.SearchField(this); });
	});
	
})(jQuery);